<?php
/**
 * Plugin Name: Dynamic Category Display
 * Plugin URI: https://digitalhubit.com
 * Description: Automatically displays all non-empty categories with customizable styling. Use shortcode [category_display] to show categories anywhere on your site.
 * Version: 1.0.0
 * Author: Tariqul Islam
 * Author URI: https://digitalhubit.com
 * License: GPL2
 * Text Domain: dynamic-category-display
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Dynamic_Category_Display {
    
    private $default_settings = array(
        'background_color' => '#F5F7FA',
        'card_background' => '#FFFFFF',
        'primary_color' => '#1B3A52',
        'accent_color' => '#00A8B5',
        'accent_hover' => '#008A95',
        'text_color' => '#2C3E50',
        'secondary_text' => '#5A6C7D',
        'border_color' => '#95A5A6',
        'section_title' => 'Explore Our Categories',
        'section_subtitle' => 'Browse through our collection',
        'show_post_count' => false,
        'exclude_categories' => array(),
        'min_posts' => 1,
        'order_by' => 'name',
        'order' => 'ASC',
        'show_empty' => false,
        'columns' => 4
    );
    
    private $cache_group = 'dynamic-category-display';
    
    private $cache_ttl = 0;
    
    private $render_cache_ttl = 0;
    
    public function __construct() {
        $this->cache_ttl = defined('HOUR_IN_SECONDS') ? HOUR_IN_SECONDS : 3600;
        $this->render_cache_ttl = defined('HOUR_IN_SECONDS') ? HOUR_IN_SECONDS * 6 : 21600;
        add_shortcode('category_display', array($this, 'render_categories'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('save_post', array($this, 'invalidate_category_cache'));
        add_action('before_delete_post', array($this, 'invalidate_category_cache'));
        add_action('edited_category', array($this, 'invalidate_category_cache_by_term'), 10, 1);
        add_action('delete_category', array($this, 'invalidate_category_cache_by_term'), 10, 1);
        add_action('created_category', array($this, 'invalidate_category_cache_by_term'), 10, 1);
        add_action('update_option_dcd_settings', array($this, 'invalidate_render_cache_on_settings'), 10, 3);
        add_action('add_option_dcd_settings', array($this, 'invalidate_render_cache'));
        add_action('switch_theme', array($this, 'invalidate_render_cache'));
        add_action('customize_save_after', array($this, 'invalidate_render_cache'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            'Category Display Settings',
            'Category Display',
            'manage_options',
            'dynamic-category-display',
            array($this, 'settings_page')
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('dcd_settings_group', 'dcd_settings');
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if ('settings_page_dynamic-category-display' !== $hook) {
            return;
        }
        
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        wp_add_inline_script(
            'wp-color-picker',
            'jQuery(function($){$(".dcd-color-picker").wpColorPicker();});'
        );
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        $settings = $this->get_settings();
        $ui_settings = $this->apply_theme_colors($settings);
        $accent_rgb = $this->hex_to_rgb($ui_settings['accent_color']);
        $accent_soft = 'rgba(' . implode(', ', array_map('intval', $accent_rgb)) . ', 0.08)';
        $theme_palette_active = $ui_settings !== $settings;
        ?>
        <div class="wrap dcd-settings-wrap" style="--dcd-primary: <?php echo esc_attr($ui_settings['primary_color']); ?>; --dcd-accent: <?php echo esc_attr($ui_settings['accent_color']); ?>; --dcd-accent-hover: <?php echo esc_attr($ui_settings['accent_hover']); ?>; --dcd-card: <?php echo esc_attr($ui_settings['card_background']); ?>; --dcd-border: <?php echo esc_attr($ui_settings['border_color']); ?>; --dcd-text: <?php echo esc_attr($ui_settings['text_color']); ?>; --dcd-background: <?php echo esc_attr($ui_settings['background_color']); ?>; --dcd-secondary: <?php echo esc_attr($ui_settings['secondary_text']); ?>; --dcd-accent-soft: <?php echo esc_attr($accent_soft); ?>;">
            <style>
                .dcd-settings-wrap {
                    max-width: 1100px;
                    margin-top: 24px;
                    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
                }
                .dcd-settings-wrap * {
                    box-sizing: border-box;
                }
                .dcd-settings-wrap h1,
                .dcd-settings-wrap h2,
                .dcd-settings-wrap h3 {
                    font-weight: 600;
                    color: #102a43;
                }
                .dcd-admin-hero {
                    position: relative;
                    overflow: hidden;
                    padding: 32px;
                    border-radius: 18px;
                    background: linear-gradient(135deg, var(--dcd-primary) 0%, var(--dcd-accent) 100%);
                    color: #ffffff;
                    box-shadow: 0 25px 50px rgba(16, 42, 67, 0.35);
                    margin-bottom: 28px;
                }
                .dcd-admin-hero::after {
                    content: "";
                    position: absolute;
                    width: 240px;
                    height: 240px;
                    background: rgba(255, 255, 255, 0.12);
                    border-radius: 50%;
                    top: -120px;
                    right: -80px;
                    filter: blur(0);
                }
                .dcd-admin-hero::before {
                    content: "";
                    position: absolute;
                    width: 320px;
                    height: 320px;
                    background: rgba(255, 255, 255, 0.08);
                    border-radius: 50%;
                    bottom: -160px;
                    left: -120px;
                }
                .dcd-admin-hero__content {
                    position: relative;
                    z-index: 1;
                    display: flex;
                    flex-direction: column;
                    gap: 8px;
                }
                .dcd-admin-hero__content h1 {
                    margin: 0;
                    font-size: 32px;
                    color: #ffffff;
                }
                .dcd-admin-hero__content p {
                    margin: 0;
                    font-size: 15px;
                    color: rgba(255, 255, 255, 0.82);
                    max-width: 520px;
                    line-height: 1.6;
                }
                .dcd-admin-hero__meta {
                    position: relative;
                    z-index: 1;
                    margin-top: 18px;
                    display: flex;
                    flex-wrap: wrap;
                    gap: 10px;
                }
                .dcd-admin-badge {
                    display: inline-flex;
                    align-items: center;
                    padding: 6px 14px;
                    border-radius: 999px;
                    background: rgba(255, 255, 255, 0.16);
                    font-size: 12px;
                    letter-spacing: 0.02em;
                    text-transform: uppercase;
                }
                .dcd-settings-form {
                    background: #ffffff;
                    border-radius: 16px;
                    padding: 28px;
                    box-shadow: 0 18px 40px rgba(16, 42, 67, 0.08);
                }
                .dcd-admin-layout {
                    display: grid;
                    grid-template-columns: minmax(0, 2fr) minmax(260px, 1fr);
                    gap: 28px;
                    align-items: start;
                }
                .dcd-panel {
                    background: #f9fbfd;
                    border: 1px solid rgba(16, 42, 67, 0.08);
                    border-radius: 14px;
                    padding: 22px;
                    box-shadow: inset 0 1px 0 rgba(255, 255, 255, 0.6);
                }
                .dcd-panel + .dcd-panel {
                    margin-top: 22px;
                }
                .dcd-panel__header {
                    margin-bottom: 18px;
                }
                .dcd-panel__header h2 {
                    margin: 0;
                    font-size: 18px;
                }
                .dcd-panel__header p {
                    margin: 8px 0 0;
                    color: #486581;
                    font-size: 13px;
                    line-height: 1.5;
                }
                .dcd-field {
                    margin-bottom: 18px;
                    display: flex;
                    flex-direction: column;
                    gap: 6px;
                }
                .dcd-field:last-of-type {
                    margin-bottom: 0;
                }
                .dcd-label {
                    font-weight: 600;
                    color: #243b53;
                    font-size: 13px;
                }
                .dcd-field input[type="text"],
                .dcd-field input[type="number"],
                .dcd-field select,
                .dcd-field textarea {
                    border-radius: 8px;
                    border: 1px solid rgba(16, 42, 67, 0.15);
                    padding: 10px 12px;
                    font-size: 14px;
                    color: #102a43;
                    background: #ffffff;
                    transition: border-color 0.2s ease, box-shadow 0.2s ease;
                }
                .dcd-field input[type="text"]:focus,
                .dcd-field input[type="number"]:focus,
                .dcd-field select:focus,
                .dcd-field textarea:focus {
                    border-color: var(--dcd-accent);
                    box-shadow: 0 0 0 1px var(--dcd-accent);
                    outline: none;
                }
                .dcd-field .description {
                    color: #627d98;
                    font-size: 12px;
                    line-height: 1.5;
                    margin: 0;
                }
                .dcd-field--inline {
                    flex-direction: row;
                    align-items: center;
                    justify-content: space-between;
                    gap: 18px;
                }
                .dcd-field--inline .dcd-field__copy {
                    max-width: 70%;
                }
                .dcd-switch {
                    position: relative;
                    display: inline-flex;
                    align-items: center;
                    cursor: pointer;
                }
                .dcd-switch input {
                    position: absolute;
                    width: 1px;
                    height: 1px;
                    padding: 0;
                    margin: -1px;
                    overflow: hidden;
                    clip: rect(0, 0, 0, 0);
                    border: 0;
                }
                .dcd-switch__track {
                    width: 46px;
                    height: 24px;
                    background: rgba(16, 42, 67, 0.2);
                    border-radius: 999px;
                    position: relative;
                    transition: background 0.2s ease;
                }
                .dcd-switch__track::after {
                    content: "";
                    position: absolute;
                    top: 3px;
                    left: 4px;
                    width: 18px;
                    height: 18px;
                    background: #ffffff;
                    border-radius: 50%;
                    box-shadow: 0 2px 6px rgba(16, 42, 67, 0.25);
                    transition: transform 0.2s ease;
                }
                .dcd-switch input:checked + .dcd-switch__track {
                    background: var(--dcd-accent);
                }
                .dcd-switch input:checked + .dcd-switch__track::after {
                    transform: translateX(20px);
                }
                .dcd-code-block {
                    display: block;
                    background: #0b1f36;
                    color: #f0f6ff;
                    padding: 12px 14px;
                    border-radius: 10px;
                    font-family: "JetBrains Mono", "Fira Code", monospace;
                    font-size: 13px;
                    margin: 12px 0;
                    word-break: break-word;
                }
                .dcd-list {
                    margin: 16px 0 0;
                    padding-left: 18px;
                    color: var(--dcd-secondary);
                    font-size: 13px;
                    line-height: 1.6;
                }
                .dcd-preview {
                    border-radius: 12px;
                    padding: 18px;
                    background: var(--dcd-accent-soft);
                }
                .dcd-preview-card {
                    border-radius: 12px;
                    border: 1px solid var(--dcd-border);
                    background: var(--dcd-card);
                    padding: 16px;
                    display: flex;
                    gap: 14px;
                    align-items: center;
                    box-shadow: 0 12px 30px rgba(15, 23, 42, 0.08);
                }
                .dcd-preview-icon {
                    width: 44px;
                    height: 44px;
                    border-radius: 10px;
                    background: linear-gradient(135deg, var(--dcd-accent) 0%, var(--dcd-accent-hover) 100%);
                    color: #ffffff;
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    font-weight: 600;
                    font-size: 12px;
                    letter-spacing: 0.08em;
                }
                .dcd-preview-meta {
                    display: flex;
                    flex-direction: column;
                    gap: 4px;
                }
                .dcd-preview-title {
                    font-size: 15px;
                    font-weight: 600;
                    color: var(--dcd-text);
                }
                .dcd-preview-count {
                    font-size: 12px;
                    color: var(--dcd-secondary);
                }
                .dcd-form-footer {
                    margin-top: 24px;
                    display: flex;
                    align-items: center;
                    gap: 14px;
                }
                .dcd-footer-note {
                    color: #627d98;
                    font-size: 12px;
                }
                .dcd-settings-wrap .notice {
                    border-radius: 10px;
                    margin: 0 0 20px;
                    box-shadow: 0 12px 30px rgba(16, 42, 67, 0.08);
                }
                .dcd-settings-wrap .notice p {
                    color: #243b53;
                    font-size: 13px;
                }
                .dcd-settings-wrap .notice p strong {
                    color: #102a43;
                }
                .dcd-settings-wrap .notice-success {
                    border-left-color: var(--dcd-accent);
                }
                @media (max-width: 1024px) {
                    .dcd-admin-layout {
                        grid-template-columns: 1fr;
                    }
                    .dcd-field--inline {
                        align-items: flex-start;
                    }
                    .dcd-field--inline .dcd-field__copy {
                        max-width: 100%;
                    }
                }
                @media (max-width: 600px) {
                    .dcd-settings-form {
                        padding: 20px;
                    }
                    .dcd-admin-hero {
                        padding: 24px;
                    }
                    .dcd-admin-hero__content h1 {
                        font-size: 26px;
                    }
                }
            </style>

            <div class="dcd-admin-hero">
                <div class="dcd-admin-hero__content">
                    <h1><?php esc_html_e('Dynamic Category Display', 'dynamic-category-display'); ?></h1>
                    <p><?php esc_html_e('Fine-tune how your categories look on the front end. Adjust styling, layout, and behavior to match your brand in just a few clicks.', 'dynamic-category-display'); ?></p>
                </div>
                <div class="dcd-admin-hero__meta">
                    <span class="dcd-admin-badge"><?php esc_html_e('Shortcode Ready', 'dynamic-category-display'); ?></span>
                    <span class="dcd-admin-badge"><?php esc_html_e('Live Style Preview', 'dynamic-category-display'); ?></span>
                    <span class="dcd-admin-badge"><?php esc_html_e('Global Settings', 'dynamic-category-display'); ?></span>
                </div>
            </div>

            <?php settings_errors(); ?>

            <form method="post" action="options.php" class="dcd-settings-form">
                <?php settings_fields('dcd_settings_group'); ?>

                <div class="dcd-admin-layout">
                    <div class="dcd-admin-main">
                        <section class="dcd-panel">
                            <div class="dcd-panel__header">
                                <h2><?php esc_html_e('Section Content', 'dynamic-category-display'); ?></h2>
                                <p><?php esc_html_e('Control the messaging that appears above your category grid.', 'dynamic-category-display'); ?></p>
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_section_title"><?php esc_html_e('Section Title', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_section_title" name="dcd_settings[section_title]" value="<?php echo esc_attr($settings['section_title']); ?>" class="regular-text" placeholder="<?php esc_attr_e('Explore Our Categories', 'dynamic-category-display'); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_section_subtitle"><?php esc_html_e('Section Subtitle', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_section_subtitle" name="dcd_settings[section_subtitle]" value="<?php echo esc_attr($settings['section_subtitle']); ?>" class="regular-text" placeholder="<?php esc_attr_e('Browse through our collection', 'dynamic-category-display'); ?>">
                                <p class="description"><?php esc_html_e('Optional supporting copy that appears below the heading.', 'dynamic-category-display'); ?></p>
                            </div>
                        </section>

                        <section class="dcd-panel">
                            <div class="dcd-panel__header">
                                <h2><?php esc_html_e('Styling', 'dynamic-category-display'); ?></h2>
                                <p><?php esc_html_e('Match colors to your brand. These colors are used for the section and category cards.', 'dynamic-category-display'); ?></p>
                                <?php if ($theme_palette_active) : ?>
                                    <p class="description"><?php esc_html_e('We detected your theme’s palette and applied it automatically. Adjust any color picker to override it.', 'dynamic-category-display'); ?></p>
                                <?php endif; ?>
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_background_color"><?php esc_html_e('Section Background', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_background_color" name="dcd_settings[background_color]" value="<?php echo esc_attr($settings['background_color']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['background_color']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_card_background"><?php esc_html_e('Card Background', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_card_background" name="dcd_settings[card_background]" value="<?php echo esc_attr($settings['card_background']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['card_background']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_primary_color"><?php esc_html_e('Primary Text Color', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_primary_color" name="dcd_settings[primary_color]" value="<?php echo esc_attr($settings['primary_color']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['primary_color']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_accent_color"><?php esc_html_e('Accent Color', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_accent_color" name="dcd_settings[accent_color]" value="<?php echo esc_attr($settings['accent_color']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['accent_color']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_accent_hover"><?php esc_html_e('Accent Hover Color', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_accent_hover" name="dcd_settings[accent_hover]" value="<?php echo esc_attr($settings['accent_hover']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['accent_hover']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_text_color"><?php esc_html_e('Body Text Color', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_text_color" name="dcd_settings[text_color]" value="<?php echo esc_attr($settings['text_color']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['text_color']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_secondary_text"><?php esc_html_e('Secondary Text Color', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_secondary_text" name="dcd_settings[secondary_text]" value="<?php echo esc_attr($settings['secondary_text']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['secondary_text']); ?>">
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_border_color"><?php esc_html_e('Border Color', 'dynamic-category-display'); ?></label>
                                <input type="text" id="dcd_border_color" name="dcd_settings[border_color]" value="<?php echo esc_attr($settings['border_color']); ?>" class="dcd-color-picker" data-default-color="<?php echo esc_attr($this->default_settings['border_color']); ?>">
                            </div>
                        </section>

                        <section class="dcd-panel">
                            <div class="dcd-panel__header">
                                <h2><?php esc_html_e('Layout & Behavior', 'dynamic-category-display'); ?></h2>
                                <p><?php esc_html_e('Configure how categories are sorted and displayed.', 'dynamic-category-display'); ?></p>
                            </div>
                            <div class="dcd-field dcd-field--inline">
                                <div class="dcd-field__copy">
                                    <span class="dcd-label"><?php esc_html_e('Show Post Count', 'dynamic-category-display'); ?></span>
                                    <p class="description"><?php esc_html_e('Display the number of posts inside each category pill.', 'dynamic-category-display'); ?></p>
                                </div>
                                <label class="dcd-switch" for="dcd_show_post_count">
                                    <input type="hidden" name="dcd_settings[show_post_count]" value="0">
                                    <input type="checkbox" id="dcd_show_post_count" name="dcd_settings[show_post_count]" value="1" <?php checked($settings['show_post_count'], 1); ?>>
                                    <span class="dcd-switch__track" aria-hidden="true"></span>
                                    <span class="screen-reader-text"><?php esc_html_e('Toggle post count display', 'dynamic-category-display'); ?></span>
                                </label>
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_min_posts"><?php esc_html_e('Minimum Posts', 'dynamic-category-display'); ?></label>
                                <input type="number" id="dcd_min_posts" name="dcd_settings[min_posts]" value="<?php echo esc_attr($settings['min_posts']); ?>" min="0" class="small-text">
                                <p class="description"><?php esc_html_e('Only show categories that meet this minimum post count.', 'dynamic-category-display'); ?></p>
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_order_by"><?php esc_html_e('Order Categories By', 'dynamic-category-display'); ?></label>
                                <select id="dcd_order_by" name="dcd_settings[order_by]">
                                    <option value="name" <?php selected($settings['order_by'], 'name'); ?>><?php esc_html_e('Name', 'dynamic-category-display'); ?></option>
                                    <option value="count" <?php selected($settings['order_by'], 'count'); ?>><?php esc_html_e('Post Count', 'dynamic-category-display'); ?></option>
                                    <option value="id" <?php selected($settings['order_by'], 'id'); ?>><?php esc_html_e('ID', 'dynamic-category-display'); ?></option>
                                </select>
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_order"><?php esc_html_e('Sort Direction', 'dynamic-category-display'); ?></label>
                                <select id="dcd_order" name="dcd_settings[order]">
                                    <option value="ASC" <?php selected($settings['order'], 'ASC'); ?>><?php esc_html_e('Ascending', 'dynamic-category-display'); ?></option>
                                    <option value="DESC" <?php selected($settings['order'], 'DESC'); ?>><?php esc_html_e('Descending', 'dynamic-category-display'); ?></option>
                                </select>
                            </div>
                            <div class="dcd-field dcd-field--inline">
                                <div class="dcd-field__copy">
                                    <span class="dcd-label"><?php esc_html_e('Show Empty Categories', 'dynamic-category-display'); ?></span>
                                    <p class="description"><?php esc_html_e('Include categories even when they do not contain any posts.', 'dynamic-category-display'); ?></p>
                                </div>
                                <label class="dcd-switch" for="dcd_show_empty">
                                    <input type="hidden" name="dcd_settings[show_empty]" value="0">
                                    <input type="checkbox" id="dcd_show_empty" name="dcd_settings[show_empty]" value="1" <?php checked($settings['show_empty'], 1); ?>>
                                    <span class="dcd-switch__track" aria-hidden="true"></span>
                                    <span class="screen-reader-text"><?php esc_html_e('Toggle empty categories visibility', 'dynamic-category-display'); ?></span>
                                </label>
                            </div>
                            <div class="dcd-field">
                                <label class="dcd-label" for="dcd_columns"><?php esc_html_e('Categories Per Row', 'dynamic-category-display'); ?></label>
                                <select id="dcd_columns" name="dcd_settings[columns]">
                                    <option value="2" <?php selected($settings['columns'], 2); ?>><?php esc_html_e('2 Columns', 'dynamic-category-display'); ?></option>
                                    <option value="3" <?php selected($settings['columns'], 3); ?>><?php esc_html_e('3 Columns', 'dynamic-category-display'); ?></option>
                                    <option value="4" <?php selected($settings['columns'], 4); ?>><?php esc_html_e('4 Columns', 'dynamic-category-display'); ?></option>
                                    <option value="5" <?php selected($settings['columns'], 5); ?>><?php esc_html_e('5 Columns', 'dynamic-category-display'); ?></option>
                                    <option value="6" <?php selected($settings['columns'], 6); ?>><?php esc_html_e('6 Columns', 'dynamic-category-display'); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e('Desktop layout setting. The grid adjusts automatically on smaller screens.', 'dynamic-category-display'); ?></p>
                            </div>
                        </section>
                    </div>

                    <aside class="dcd-admin-sidebar">
                        <section class="dcd-panel">
                            <div class="dcd-panel__header">
                                <h2><?php esc_html_e('Shortcode Quick Start', 'dynamic-category-display'); ?></h2>
                                <p><?php esc_html_e('Copy and paste the shortcode anywhere you need a dynamic category grid.', 'dynamic-category-display'); ?></p>
                            </div>
                            <span class="dcd-code-block">[category_display]</span>
                            <ul class="dcd-list">
                                <li><code>[category_display title=&quot;Custom Title&quot;]</code></li>
                                <li><code>[category_display subtitle=&quot;Custom Subtitle&quot;]</code></li>
                                <li><code>[category_display exclude=&quot;1,5,10&quot;]</code></li>
                                <li><code>[category_display min_posts=&quot;5&quot;]</code></li>
                                <li><code>[category_display order_by=&quot;count&quot;]</code></li>
                                <li><code>[category_display columns=&quot;3&quot;]</code></li>
                            </ul>
                        </section>

                        <section class="dcd-panel">
                            <div class="dcd-panel__header">
                                <h2><?php esc_html_e('Live Style Preview', 'dynamic-category-display'); ?></h2>
                                <p><?php esc_html_e('See a quick snapshot of how the current palette will render.', 'dynamic-category-display'); ?></p>
                            </div>
                            <div class="dcd-preview">
                                <div class="dcd-preview-card">
                                    <span class="dcd-preview-icon"><?php esc_html_e('CAT', 'dynamic-category-display'); ?></span>
                                    <div class="dcd-preview-meta">
                                        <span class="dcd-preview-title"><?php esc_html_e('Sample Category', 'dynamic-category-display'); ?></span>
                                        <?php if ($settings['show_post_count']) : ?>
                                            <span class="dcd-preview-count"><?php esc_html_e('8 posts', 'dynamic-category-display'); ?></span>
                                        <?php else : ?>
                                            <span class="dcd-preview-count"><?php esc_html_e('Post count hidden', 'dynamic-category-display'); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </section>
                    </aside>
                </div>

                <div class="dcd-form-footer">
                    <?php submit_button(esc_html__('Save Settings', 'dynamic-category-display'), 'primary button-hero dcd-save-button', 'submit', false); ?>
                    <span class="dcd-footer-note"><?php esc_html_e('Changes apply instantly to any page using the shortcode.', 'dynamic-category-display'); ?></span>
                </div>
            </form>
        </div>
        <?php
    }

    /**
     * Get settings
     */
    private function get_settings() {
        $saved_settings = get_option('dcd_settings', array());
        return wp_parse_args($saved_settings, $this->default_settings);
    }
    
    /**
     * Retrieve posts for a category with caching
     */
    private function get_cached_category_posts($category_id) {
        $category_id = intval($category_id);
        if ($category_id <= 0) {
            return array();
        }
        
        $cache_key = 'dcd_posts_' . $category_id;
        $cached = wp_cache_get($cache_key, $this->cache_group);
        if (false !== $cached) {
            return $cached;
        }
        
        $posts = get_posts(array(
            'category' => $category_id,
            'numberposts' => -1,
            'post_status' => 'publish',
            'orderby' => 'date',
            'order' => 'DESC',
            'no_found_rows' => true,
            'ignore_sticky_posts' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
            'suppress_filters' => true,
        ));
        
        wp_cache_set($cache_key, $posts, $this->cache_group, $this->cache_ttl);
        
        return $posts;
    }
    
    /**
     * Invalidate cached posts for categories related to a post
     */
    public function invalidate_category_cache($post_id) {
        $post_id = intval($post_id);
        
        if ($post_id <= 0 || (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)) {
            return;
        }
        
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }
        
        $post_type = get_post_type($post_id);
        if ('post' !== $post_type) {
            return;
        }
        
        $category_ids = wp_get_post_categories($post_id);
        if (empty($category_ids)) {
            return;
        }
        
        foreach ($category_ids as $category_id) {
            wp_cache_delete('dcd_posts_' . intval($category_id), $this->cache_group);
        }
        
        $this->bump_cache_version();
    }
    
    /**
     * Invalidate cached posts for a specific category
     */
    public function invalidate_category_cache_by_term($term_id) {
        $term_id = intval($term_id);
        if ($term_id <= 0) {
            return;
        }
        
        wp_cache_delete('dcd_posts_' . $term_id, $this->cache_group);
        $this->bump_cache_version();
    }
    
    /**
     * Retrieve current cache version
     */
    private function get_cache_version() {
        $version = get_option('dcd_cache_version');
        
        if (!$version) {
            $version = microtime(true);
            update_option('dcd_cache_version', $version, false);
        }
        
        return $version;
    }
    
    /**
     * Increment cache version to invalidate render cache
     */
    private function bump_cache_version() {
        update_option('dcd_cache_version', microtime(true), false);
    }
    
    /**
     * Build a unique cache key for the rendered section
     */
    private function get_render_cache_key($atts, $settings) {
        $signature = array(
            'atts'      => $atts,
            'settings'  => array(
                'background_color' => $settings['background_color'],
                'card_background'  => $settings['card_background'],
                'primary_color'    => $settings['primary_color'],
                'accent_color'     => $settings['accent_color'],
                'accent_hover'     => $settings['accent_hover'],
                'text_color'       => $settings['text_color'],
                'secondary_text'   => $settings['secondary_text'],
                'border_color'     => $settings['border_color'],
                'show_post_count'  => $settings['show_post_count'],
            ),
            'theme'     => get_stylesheet(),
            'locale'    => get_locale(),
            'version'   => $this->get_cache_version(),
        );
        
        $encoded = function_exists('wp_json_encode') ? wp_json_encode($signature) : json_encode($signature);
        
        return 'render_' . md5((string) $encoded);
    }
    
    /**
     * Handle render cache invalidation when settings change
     */
    public function invalidate_render_cache_on_settings($old_value, $value, $option) {
        $this->invalidate_render_cache();
    }
    
    /**
     * Bump cache version for full render cache
     */
    public function invalidate_render_cache($context = null) {
        $this->bump_cache_version();
    }
    
    /**
     * Apply theme-derived colors when defaults are in use
     */
    private function apply_theme_colors($settings) {
        $palette = $this->get_theme_palette();
        
        if (empty($palette)) {
            return $settings;
        }
        
        $primary = $this->pick_palette_color($palette, array('primary', 'foreground', 'contrast'));
        if ($primary && $this->is_default_setting('primary_color', $settings)) {
            $settings['primary_color'] = $primary;
        }
        
        $text = $this->pick_palette_color($palette, array('foreground', 'contrast', 'primary'));
        if ($text && $this->is_default_setting('text_color', $settings)) {
            $settings['text_color'] = $text;
        }
        
        $secondary = $text ? $this->adjust_color_luminance($text, 35) : null;
        if ($secondary && $this->is_default_setting('secondary_text', $settings)) {
            $settings['secondary_text'] = $secondary;
        }
        
        $background = $this->pick_palette_color($palette, array('background', 'base', 'surface'));
        if ($background && $this->is_default_setting('background_color', $settings)) {
            $settings['background_color'] = $background;
        }
        
        if ($background && $this->is_default_setting('card_background', $settings)) {
            $settings['card_background'] = $this->adjust_color_luminance($background, 6);
        }
        
        $accent = $this->pick_palette_color($palette, array('accent', 'secondary', 'primary'));
        if ($accent && $this->is_default_setting('accent_color', $settings)) {
            $settings['accent_color'] = $accent;
        }
        
        if (!empty($settings['accent_color']) && $this->is_default_setting('accent_hover', $settings)) {
            $settings['accent_hover'] = $this->adjust_color_luminance($settings['accent_color'], -12);
        }
        
        if (!empty($settings['accent_color']) && $this->is_default_setting('border_color', $settings)) {
            $settings['border_color'] = $this->adjust_color_luminance($settings['accent_color'], 60);
        }
        
        return $settings;
    }
    
    /**
     * Determine if a setting still uses the default value
     */
    private function is_default_setting($key, $settings) {
        if (!array_key_exists($key, $settings) || !array_key_exists($key, $this->default_settings)) {
            return false;
        }
        
        return $settings[$key] === $this->default_settings[$key];
    }
    
    /**
     * Retrieve theme palette approximations
     */
    private function get_theme_palette() {
        $palette = array();
        
        if (function_exists('wp_get_global_settings')) {
            $theme_palette = wp_get_global_settings(array('color', 'palette', 'theme'));
            
            if (empty($theme_palette)) {
                $theme_palette = wp_get_global_settings(array('color', 'palette', 'default'));
            }
            
            if (is_array($theme_palette)) {
                foreach ($theme_palette as $entry) {
                    if (empty($entry['slug']) || empty($entry['color'])) {
                        continue;
                    }
                    $normalized = $this->normalize_hex_color($entry['color']);
                    if ($normalized) {
                        $palette[$entry['slug']] = $normalized;
                    }
                }
            }
        }
        
        if (empty($palette)) {
            $background = get_theme_mod('background_color');
            if ($background) {
                $background = $this->normalize_hex_color($background);
                if ($background) {
                    $palette['background'] = $background;
                }
            }
            
            $header_text = get_theme_mod('header_textcolor');
            if ($header_text && 'blank' !== $header_text) {
                $header_text = $this->normalize_hex_color($header_text);
                if ($header_text) {
                    $palette['foreground'] = $header_text;
                }
            }
            
            $link_color = get_theme_mod('link_color');
            if ($link_color) {
                $link_color = $this->normalize_hex_color($link_color);
                if ($link_color) {
                    $palette['primary'] = $link_color;
                }
            }
            
            if (empty($palette['accent'])) {
                $accent_candidates = array('accent_color', 'primary_color', 'theme_accent_color', 'button_color');
                foreach ($accent_candidates as $candidate) {
                    $candidate_value = get_theme_mod($candidate);
                    if ($candidate_value) {
                        $candidate_value = $this->normalize_hex_color($candidate_value);
                        if ($candidate_value) {
                            $palette['accent'] = $candidate_value;
                            break;
                        }
                    }
                }
            }
            
            if (empty($palette['secondary'])) {
                $secondary_candidates = array('secondary_color', 'text_color');
                foreach ($secondary_candidates as $candidate) {
                    $candidate_value = get_theme_mod($candidate);
                    if ($candidate_value) {
                        $candidate_value = $this->normalize_hex_color($candidate_value);
                        if ($candidate_value) {
                            $palette['secondary'] = $candidate_value;
                            break;
                        }
                    }
                }
            }
        }
        
        return array_filter($palette);
    }
    
    /**
     * Helper to pick first available color
     */
    private function pick_palette_color($palette, $options) {
        foreach ((array) $options as $option) {
            if (!empty($palette[$option])) {
                return $palette[$option];
            }
        }
        return null;
    }
    
    /**
     * Normalize potential hex colors
     */
    private function normalize_hex_color($color) {
        if (empty($color)) {
            return '';
        }
        
        $color = trim($color);
        if ($color === '') {
            return '';
        }
        if ($color[0] !== '#') {
            $color = '#' . $color;
        }
        
        if (function_exists('sanitize_hex_color')) {
            $sanitized = sanitize_hex_color($color);
            if ($sanitized) {
                return $sanitized;
            }
        }
        
        // Manual fallback sanitization
        $color = ltrim($color, '#');
        if (strlen($color) === 3 || strlen($color) === 6) {
            if (ctype_xdigit($color)) {
                return '#' . $color;
            }
        }
        
        return '';
    }
    
    /**
     * Convert HEX color to RGB array
     */
    private function hex_to_rgb($hex) {
        $hex = ltrim($hex, '#');
        
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        
        if (strlen($hex) !== 6) {
            return array(0, 0, 0);
        }
        
        return array(
            hexdec(substr($hex, 0, 2)),
            hexdec(substr($hex, 2, 2)),
            hexdec(substr($hex, 4, 2))
        );
    }
    
    /**
     * Convert RGB array to HEX string
     */
    private function rgb_to_hex($rgb) {
        if (!is_array($rgb) || count($rgb) !== 3) {
            return '#000000';
        }
        
        return sprintf('#%02x%02x%02x', max(0, min(255, intval($rgb[0]))), max(0, min(255, intval($rgb[1]))), max(0, min(255, intval($rgb[2]))));
    }
    
    /**
     * Lighten or darken a hex color
     */
    private function adjust_color_luminance($hex, $percent) {
        $normalized = $this->normalize_hex_color($hex);
        if (!$normalized) {
            return $hex;
        }
        
        $rgb = $this->hex_to_rgb($normalized);
        $percent = max(-100, min(100, $percent));
        
        foreach ($rgb as &$component) {
            if ($percent >= 0) {
                $component = $component + (255 - $component) * ($percent / 100);
            } else {
                $component = $component + $component * ($percent / 100);
            }
            $component = max(0, min(255, round($component)));
        }
        unset($component);
        
        return $this->rgb_to_hex($rgb);
    }
    
    /**
     * Normalize title casing for post labels
     */
    private function format_post_title($title) {
        $title = trim($title);
        
        if ($title === '') {
            return $title;
        }
        
        if (function_exists('mb_convert_case')) {
            return mb_convert_case($title, MB_CASE_TITLE, 'UTF-8');
        }
        
        return ucwords(strtolower($title));
    }
    
    /**
     * Get category icon based on name
     */
    private function get_category_icon($category_name) {
        $category_name = (string) $category_name;
        $name_lower = strtolower($category_name);

        $icon_map = array(
            array(
                'icon' => '&#128267;',
                'keywords' => array('electronics', 'electronic', 'gadget', 'gadgets', 'device', 'devices', 'smart device', 'smart devices', 'tech accessory', 'tech accessories')
            ),
            array(
                'icon' => '&#128187;',
                'keywords' => array('laptop', 'laptops', 'notebook', 'notebooks', 'ultrabook', 'chromebook', 'macbook')
            ),
            array(
                'icon' => '&#128421;',
                'keywords' => array('computer', 'computers', 'desktop', 'desktops', 'pc', 'pc gaming', 'workstation', 'monitor', 'monitors', 'gaming monitor', 'gaming monitors', 'all in one')
            ),
            array(
                'icon' => '&#9881;',
                'keywords' => array('component', 'components', 'gpu', 'graphics card', 'cpu', 'motherboard', 'ram', 'memory', 'ssd', 'hdd', 'storage drive', 'peripheral', 'peripherals', 'keyboard', 'keyboards', 'mouse', 'mice', 'trackpad', 'printer', 'printers', 'scanner', 'scanners', 'webcam', 'webcams', 'router', 'routers', 'modem', 'hub', 'docking station', 'projector', 'projectors', 'calculator', 'calculators', 'shredder', 'shredders')
            ),
            array(
                'icon' => '&#128247;',
                'keywords' => array('camera', 'cameras', 'dslr', 'mirrorless', 'lens', 'lenses', 'tripod', 'tripods', 'lighting kit', 'photo accessory', 'photo accessories')
            ),
            array(
                'icon' => '&#128274;',
                'keywords' => array('security camera', 'security cameras', 'home surveillance', 'surveillance', 'smart camera', 'smart cameras', 'dash cam', 'dash cams', 'doorbell camera', 'home security', 'vehicle gps')
            ),
            array(
                'icon' => '&#128241;',
                'keywords' => array('smartphone', 'smartphones', 'cell phone', 'cell phones', 'mobile phone', 'mobile phones', 'phone', 'phones', 'phone case', 'phone cases', 'screen protector', 'screen protectors', 'charger', 'chargers', 'power bank', 'power banks')
            ),
            array(
                'icon' => '&#128250;',
                'keywords' => array('television', 'televisions', 'tv', 'smart tv', 'projector', 'projectors', 'streaming device', 'streaming devices', 'blu-ray', 'blu ray', 'sound bar', 'sound bars', 'speaker', 'speakers', 'receiver', 'receivers', 'home theater', 'fire tv', 'echo', 'echo device', 'fire tablet', 'kindle', 'bluetooth speaker', 'bluetooth speakers')
            ),
            array(
                'icon' => '&#127911;',
                'keywords' => array('headphone', 'headphones', 'earbud', 'earbuds', 'audio', 'sound', 'music device', 'sound bar', 'soundbar', 'speaker system')
            ),
            array(
                'icon' => '&#8986;',
                'keywords' => array('smartwatch', 'smartwatches', 'fitness tracker', 'fitness trackers', 'wearable', 'wearables', 'wearable camera', 'wearable cameras', 'vr headset', 'vr headsets')
            ),
            array(
                'icon' => '&#127918;',
                'keywords' => array('gaming', 'game', 'video game', 'video games', 'console', 'playstation', 'xbox', 'nintendo switch', 'pc game', 'pc games', 'controller', 'controllers', 'gaming headset', 'gaming headsets', 'retro game', 'retro games')
            ),
            array(
                'icon' => '&#128663;',
                'keywords' => array('car audio', 'car electronics', 'car accessory', 'car accessories', 'car part', 'car parts', 'batteries', 'tires', 'rims', 'car cover', 'car covers', 'roof rack', 'roof racks', 'seat cover', 'seat covers', 'garage tool', 'garage tools', 'motorcycle part', 'motorcycle parts', 'detailing supplies')
            ),
            array(
                'icon' => '&#127968;',
                'keywords' => array('refrigerator', 'refrigerators', 'freezer', 'freezers', 'washer', 'washers', 'dryer', 'dryers', 'oven', 'ovens', 'cooktop', 'cooktops', 'dishwasher', 'dishwashers', 'microwave', 'microwaves', 'appliance', 'appliances', 'home appliance')
            ),
            array(
                'icon' => '&#129529;',
                'keywords' => array('vacuum', 'vacuums', 'robot vacuum', 'robot vacuums', 'carpet cleaner', 'carpet cleaners', 'air conditioner', 'air conditioners', 'heater', 'heaters', 'air purifier', 'air purifiers', 'humidifier', 'humidifiers', 'cleaning supply', 'cleaning supplies', 'laundry organizer', 'storage bin', 'storage bins', 'organizer', 'organizers')
            ),
            array(
                'icon' => '&#127859;',
                'keywords' => array('blender', 'blenders', 'mixer', 'mixers', 'coffee maker', 'coffee makers', 'toaster', 'toasters', 'air fryer', 'air fryers', 'kitchen gadget', 'kitchen gadgets', 'slow cooker', 'instant pot')
            ),
            array(
                'icon' => '&#127858;',
                'keywords' => array('cookware', 'bakeware', 'dinnerware', 'utensil', 'utensils', 'kitchen tool', 'kitchen tools', 'kitchenware')
            ),
            array(
                'icon' => '&#128715;',
                'keywords' => array('furniture', 'living room furniture', 'bedroom furniture', 'office furniture', 'patio furniture', 'home furniture', 'sofa', 'couch', 'desk', 'table', 'chair', 'chairs', 'bedding', 'sheets', 'pillows', 'towels', 'home decor', 'wall art', 'rug', 'rugs', 'decor', 'home accents')
            ),
            array(
                'icon' => '&#128161;',
                'keywords' => array('lamp', 'lamps', 'ceiling light', 'ceiling lights', 'lighting', 'light fixture', 'smart bulb', 'smart bulbs', 'outdoor lighting')
            ),
            array(
                'icon' => '&#128296;',
                'keywords' => array('power tool', 'power tools', 'hand tool', 'hand tools', 'drill', 'drills', 'saw', 'saws', 'hardware', 'locks', 'lock', 'door knob', 'door knobs', 'fastener', 'fasteners', 'faucet', 'faucets', 'sink', 'sinks', 'showerhead', 'showerheads', 'toilet', 'toilets', 'electrical supply', 'plumbing supply', 'plumbing supplies', 'electrical supplies')
            ),
            array(
                'icon' => '&#128268;',
                'keywords' => array('smart home', 'smart home device', 'smart home devices', 'thermostat', 'thermostats', 'smoke detector', 'smoke detectors', 'smart lock', 'smart locks', 'smart thermostat', 'smart camera', 'smart cameras', 'smart bulb', 'smart bulbs', 'home automation', 'smart security')
            ),
            array(
                'icon' => '&#127795;',
                'keywords' => array('patio', 'patio furniture', 'lawn mower', 'lawn mowers', 'trimmer', 'trimmers', 'gardening tool', 'gardening tools', 'garden', 'outdoor furniture', 'grill', 'grills', 'smoker', 'smokers', 'outdoor game', 'outdoor games', 'pool', 'pools', 'hot tub', 'hot tubs', 'outdoor gear')
            ),
            array(
                'icon' => '&#128132;',
                'keywords' => array('makeup', 'cosmetic', 'cosmetics', 'skincare', 'skin care', 'haircare', 'hair care', 'fragrance', 'perfume', 'nail care', 'grooming tool', 'grooming tools', 'shaving kit', 'personal care', 'oral care', 'luxury beauty', 'grooming')
            ),
            array(
                'icon' => '&#128137;',
                'keywords' => array('vitamin', 'vitamins', 'supplement', 'supplements', 'health monitor', 'health monitors', 'medical equipment', 'thermometer', 'thermometers', 'first aid', 'health', 'wellness', 'medical', 'healthcare')
            ),
            array(
                'icon' => '&#128118;',
                'keywords' => array('baby', 'babies', 'diaper', 'diapers', 'wipes', 'baby monitor', 'baby monitors', 'baby furniture', 'crib', 'cribs', 'stroller', 'strollers', 'car seat', 'car seats', 'high chair', 'high chairs', 'baby toy', 'baby toys', 'baby bath', 'baby baths', 'baby lotion', 'baby lotions', 'baby apparel')
            ),
            array(
                'icon' => '&#127869;',
                'keywords' => array('organic food', 'organic foods', 'snack', 'snacks', 'beverage', 'beverages', 'coffee', 'tea', 'candy', 'pantry', 'pantry staples', 'spices', 'gluten free', 'gourmet gift', 'gourmet gifts', 'food', 'groceries')
            ),
            array(
                'icon' => '&#128089;',
                'keywords' => array('womens clothing', 'mens clothing', 'kids clothing', 'baby apparel', 'clothing', 'apparel', 'shoes', 'handbag', 'handbags', 'jewelry', 'watches', 'accessories', 'luxury fashion', 'designer', 'designer accessories', 'high-end jewelry')
            ),
            array(
                'icon' => '&#128747;',
                'keywords' => array('suitcase', 'suitcases', 'carry on', 'carry ons', 'carry-on', 'backpack', 'backpacks', 'duffel bag', 'duffel bags', 'travel pillow', 'travel pillows', 'passport holder', 'travel gear', 'luggage')
            ),
            array(
                'icon' => '&#127947;',
                'keywords' => array('exercise equipment', 'fitness equipment', 'dumbbell', 'dumbbells', 'yoga mat', 'yoga mats', 'resistance band', 'resistance bands', 'camping gear', 'hiking backpack', 'hiking backpacks', 'tent', 'tents', 'sleeping bag', 'sleeping bags', 'fishing gear', 'hunting equipment', 'bicycle', 'bicycles', 'bike accessory', 'bike accessories', 'team sports gear', 'water sports', 'golf equipment', 'sports apparel', 'sports memorabilia', 'fitness')
            ),
            array(
                'icon' => '&#9874;',
                'keywords' => array('industrial supplies', 'lab equipment', 'safety gear', 'cleaning chemical', 'cleaning chemicals', 'industrial', 'laboratory')
            ),
            array(
                'icon' => '&#128396;',
                'keywords' => array('art supplies', 'paint', 'paints', 'brush', 'brushes', 'canvas', 'canvases', 'craft kit', 'craft kits', 'sewing machine', 'sewing machines', 'yarn', 'fabrics', 'fabric', 'crochet', 'crafting', 'handmade furniture', 'handmade jewelry')
            ),
            array(
                'icon' => '&#127928;',
                'keywords' => array('musical instrument', 'musical instruments', 'guitar', 'guitars', 'keyboard', 'keyboards', 'drum', 'drums', 'dj equipment', 'recording gear')
            ),
            array(
                'icon' => '&#128206;',
                'keywords' => array('office supply', 'office supplies', 'paper', 'pens', 'ink', 'toner', 'office chair', 'office chairs', 'desk', 'desks', 'notebook', 'notebooks', 'school supplies')
            ),
            array(
                'icon' => '&#128054;',
                'keywords' => array('pet food', 'dog toy', 'dog toys', 'cat litter', 'fish tank', 'bird cage', 'small animal supplies', 'pet grooming', 'pet clothing', 'pet accessories', 'pet supplies')
            ),
            array(
                'icon' => '&#128142;',
                'keywords' => array('coins', 'fine art', 'antique', 'antiques', 'stamp', 'stamps', 'collectible', 'collectibles', 'movie memorabilia', 'comic book', 'comic books', 'vinyl record', 'vinyl records', 'luxury', 'luxury beauty', 'luxury fashion', 'designer accessories', 'high end jewelry')
            ),
            array(
                'icon' => '&#128190;',
                'keywords' => array('office software', 'mobile app', 'mobile apps', 'mobile game', 'mobile games', 'antivirus', 'graphic design software', 'operating system', 'educational software', 'software', 'app', 'apps', 'subscription box', 'subscription boxes')
            ),
            array(
                'icon' => '&#128218;',
                'keywords' => array('book', 'books', 'ebook', 'ebooks', 'fiction', 'nonfiction', 'children books', 'textbook', 'textbooks', 'comic', 'comics', 'magazine', 'magazines')
            ),
            array(
                'icon' => '&#127916;',
                'keywords' => array('dvd', 'dvds', 'blu-ray', 'blu-rays', 'tv box set', 'tv box sets', 'cd', 'cds', 'mp3', 'mp3 downloads', 'movie', 'movies', 'tv series', 'streaming', 'media')
            )
        );

        foreach ($icon_map as $mapping) {
            foreach ($mapping['keywords'] as $keyword) {
                $keyword = trim($keyword);
                if ($keyword === '') {
                    continue;
                }

                if (strpos($name_lower, $keyword) !== false) {
                    return apply_filters('dcd_category_icon', $mapping['icon'], $category_name);
                }
            }
        }

        $fallback = $this->get_icon_fallback($category_name);

        return apply_filters('dcd_category_icon', $fallback, $category_name);
    }

    private function get_icon_fallback($category_name) {
        $trimmed = trim((string) $category_name);

        if ($trimmed === '') {
            return '&#9733;';
        }

        if (function_exists('mb_substr')) {
            $letter = mb_strtoupper(mb_substr($trimmed, 0, 1, 'UTF-8'), 'UTF-8');
        } else {
            $letter = strtoupper(substr($trimmed, 0, 1));
        }

        if (function_exists('esc_html')) {
            return esc_html($letter);
        }

        return htmlspecialchars($letter, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Render categories shortcode
     */
    public function render_categories($atts) {
        $settings = $this->get_settings();
        $settings = $this->apply_theme_colors($settings);
        
        // Parse shortcode attributes
        $atts = shortcode_atts(array(
            'title' => $settings['section_title'],
            'subtitle' => $settings['section_subtitle'],
            'exclude' => '',
            'min_posts' => $settings['min_posts'],
            'order_by' => $settings['order_by'],
            'order' => $settings['order'],
            'show_empty' => $settings['show_empty'],
            'columns' => $settings['columns']
        ), $atts);

        $atts['show_empty'] = (bool) intval($atts['show_empty']);
        $atts['min_posts'] = max(0, intval($atts['min_posts']));
        $atts['columns'] = intval($atts['columns']);
        $atts['exclude'] = trim((string) $atts['exclude']);
        $atts['order_by'] = in_array($atts['order_by'], array('name', 'count', 'id'), true) ? $atts['order_by'] : 'name';
        $atts['order'] = strtoupper($atts['order']);
        $atts['order'] = in_array($atts['order'], array('ASC', 'DESC'), true) ? $atts['order'] : 'ASC';

        if ($atts['columns'] < 2) {
            $atts['columns'] = 2;
        } elseif ($atts['columns'] > 6) {
            $atts['columns'] = 6;
        }

        $cache_key = $this->get_render_cache_key($atts, $settings);
        $cached_output = wp_cache_get($cache_key, $this->cache_group);
        if (false !== $cached_output) {
            return $cached_output;
        }

        // Get categories
        $args = array(
            'orderby' => $atts['order_by'],
            'order' => $atts['order'],
            'hide_empty' => !$atts['show_empty'],
            'exclude' => $atts['exclude'],
            'pad_counts' => false,
            'hierarchical' => false,
            'update_term_meta_cache' => false,
        );
        
        $categories = get_categories($args);
        
        // Remove default uncategorized buckets
        $categories = array_filter($categories, function($cat) {
            $name = strtolower($cat->name);
            $slug = strtolower($cat->slug);
            return !in_array($name, array('uncategorized', 'uncategorised'), true) && !in_array($slug, array('uncategorized', 'uncategorised'), true);
        });
        
        // Filter by minimum posts when empty categories are hidden
        if (!$atts['show_empty'] && $atts['min_posts'] > 0) {
            $categories = array_filter($categories, function($cat) use ($atts) {
                return $cat->count >= $atts['min_posts'];
            });
        }
        
        // Columns sanitized above
        $columns = (int) $atts['columns'];
        
        // Calculate min width based on columns
        $min_width = floor(1200 / $columns) - 20; // 1200px max width, 20px gap
        
        // Return empty if no categories
        if (empty($categories)) {
            wp_cache_set($cache_key, '', $this->cache_group, $this->render_cache_ttl);
            return '';
        }
        
        // Start output
        ob_start();
        ?>
        
        <section class="dcd-category-section">
            <style>
                .dcd-category-section {
                    padding: 60px 20px;
                    background: <?php echo esc_attr($settings['background_color']); ?>;
                    position: relative;
                }
                
                .dcd-categories-container {
                    max-width: 1200px;
                    margin: 0 auto;
                    position: relative;
                }
                
                .dcd-categories-header {
                    text-align: center;
                    margin-bottom: 50px;
                }
                
                .dcd-categories-header h2 {
                    font-size: 32px;
                    font-weight: 700;
                    color: <?php echo esc_attr($settings['primary_color']); ?>;
                    margin-bottom: 10px;
                    letter-spacing: -0.5px;
                }
                
                .dcd-categories-header p {
                    font-size: 16px;
                    color: <?php echo esc_attr($settings['secondary_text']); ?>;
                    font-weight: 400;
                }
                
                .dcd-category-grid {
                    display: grid;
                    grid-template-columns: repeat(<?php echo esc_attr($columns); ?>, 1fr);
                    gap: 20px;
                    list-style: none;
                    margin: 0;
                    padding: 0;
                    overflow: visible;
                }
                
                @media (max-width: 1024px) {
                    .dcd-category-grid {
                        grid-template-columns: repeat(<?php echo min(3, $columns); ?>, 1fr);
                    }
                }
                
                @media (max-width: 768px) {
                    .dcd-category-grid {
                        grid-template-columns: repeat(2, 1fr);
                        gap: 15px;
                    }
                }
                
                @media (max-width: 480px) {
                    .dcd-category-grid {
                        grid-template-columns: 1fr;
                    }
                }
                
                .dcd-category-item {
                    background: <?php echo esc_attr($settings['card_background']); ?>;
                    border-radius: 10px;
                    overflow: visible;
                    transition: all 0.3s cubic-bezier(0.165, 0.84, 0.44, 1);
                    box-shadow: 0 2px 8px rgba(27, 58, 82, 0.08);
                    border: 1px solid <?php echo esc_attr($settings['border_color']); ?>;
                    position: relative;
                    z-index: 1;
                }
                
                .dcd-category-item::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    height: 3px;
                    background: linear-gradient(90deg, <?php echo esc_attr($settings['accent_color']); ?> 0%, <?php echo esc_attr($settings['accent_hover']); ?> 100%);
                    transform: scaleX(0);
                    transition: transform 0.3s ease;
                }
                
                .dcd-category-item:hover {
                    transform: translateY(-5px);
                    box-shadow: 0 8px 20px rgba(27, 58, 82, 0.15);
                    border-color: <?php echo esc_attr($settings['accent_color']); ?>;
                    z-index: 40;
                }
                
                .dcd-category-item:focus-within {
                    z-index: 40;
                }
                
                .dcd-category-item:hover::before {
                    transform: scaleX(1);
                }
                
                .dcd-category-link {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 18px 20px;
                    text-decoration: none;
                    color: <?php echo esc_attr($settings['text_color']); ?>;
                    transition: color 0.3s ease;
                }
                
                .dcd-category-link:hover {
                    color: <?php echo esc_attr($settings['accent_color']); ?>;
                }
                
                .dcd-category-icon {
                    width: 36px;
                    height: 36px;
                    background: linear-gradient(135deg, <?php echo esc_attr($settings['accent_color']); ?> 0%, <?php echo esc_attr($settings['accent_hover']); ?> 100%);
                    border-radius: 8px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    margin-right: 12px;
                    font-size: 18px;
                    color: #FFFFFF;
                    flex-shrink: 0;
                }
                
                .dcd-category-content {
                    flex-grow: 1;
                }
                
                .dcd-category-name {
                    display: block;
                    font-size: 15px;
                    font-weight: 600;
                    margin-bottom: 3px;
                }
                
                .dcd-category-count {
                    font-size: 12px;
                    color: <?php echo esc_attr($settings['secondary_text']); ?>;
                    font-weight: 400;
                }
                
                .dcd-category-arrow {
                    font-size: 16px;
                    color: <?php echo esc_attr($settings['border_color']); ?>;
                    transition: all 0.3s ease;
                }
                
                .dcd-category-item:hover .dcd-category-arrow {
                    color: <?php echo esc_attr($settings['accent_color']); ?>;
                    transform: translateX(4px);
                }
                
                .dcd-category-posts {
                    position: absolute;
                    top: calc(100% - 12px);
                    left: 0;
                    right: 0;
                    background: <?php echo esc_attr($settings['card_background']); ?>;
                    border: 1px solid <?php echo esc_attr($settings['border_color']); ?>;
                    border-radius: 14px;
                    padding: 18px 20px 20px;
                    box-shadow: 0 16px 35px rgba(15, 23, 42, 0.12);
                    opacity: 0;
                    transform: translateY(32px);
                    pointer-events: none;
                    transition: opacity 0.25s ease, transform 0.25s ease;
                    z-index: 15;
                    max-height: 240px;
                    overflow-y: auto;
                }
                
                .dcd-category-posts::-webkit-scrollbar {
                    width: 8px;
                }
                
                .dcd-category-posts::-webkit-scrollbar-thumb {
                    background: <?php echo esc_attr($settings['accent_color']); ?>;
                    border-radius: 999px;
                }
                
                .dcd-category-posts::-webkit-scrollbar-track {
                    background: rgba(15, 23, 42, 0.06);
                }
                
                .dcd-category-item:hover .dcd-category-posts,
                .dcd-category-item:focus-within .dcd-category-posts {
                    opacity: 1;
                    transform: translateY(12px);
                    pointer-events: auto;
                }
                
                .dcd-category-posts__title {
                    display: block;
                    font-size: 13px;
                    font-weight: 600;
                    color: <?php echo esc_attr($settings['secondary_text']); ?>;
                    text-transform: uppercase;
                    letter-spacing: 0.04em;
                    margin-bottom: 12px;
                }
                
                .dcd-posts-list {
                    margin: 0;
                    padding: 0;
                    list-style: none;
                    display: flex;
                    flex-direction: column;
                    gap: 8px;
                }
                
                .dcd-posts-list__item {
                    margin: 0;
                }
                
                .dcd-posts-list__link {
                    display: block;
                    padding: 8px 10px;
                    border-radius: 8px;
                    text-decoration: none;
                    color: <?php echo esc_attr($settings['text_color']); ?>;
                    font-size: 14px;
                    line-height: 1.4;
                    transition: background 0.2s ease, color 0.2s ease;
                }
                
                .dcd-posts-list__link:hover,
                .dcd-posts-list__link:focus {
                    background: rgba(<?php echo implode(', ', $this->hex_to_rgb($settings['accent_color'])); ?>, 0.08);
                    color: <?php echo esc_attr($settings['accent_color']); ?>;
                    outline: none;
                }
                
                .dcd-category-posts__empty {
                    font-size: 13px;
                    color: <?php echo esc_attr($settings['secondary_text']); ?>;
                }
                
                @media (max-width: 768px) {
                    .dcd-category-section {
                        padding: 40px 15px;
                    }
                    
                    .dcd-categories-header h2 {
                        font-size: 26px;
                    }
                    
                    .dcd-category-item {
                        overflow: hidden;
                    }
                    
                    .dcd-category-posts {
                        position: relative;
                        top: auto;
                        left: auto;
                        right: auto;
                        transform: none !important;
                        opacity: 1;
                        pointer-events: auto;
                        margin-top: 16px;
                        max-height: 200px;
                    }
                    
                    .dcd-category-item:not(:hover) .dcd-category-posts {
                        opacity: 1;
                    }
                }
            </style>
            
            <article class="dcd-categories-container">
                <header class="dcd-categories-header">
                    <h2><?php echo esc_html($atts['title']); ?></h2>
                    <?php if (!empty($atts['subtitle'])) : ?>
                        <p><?php echo esc_html($atts['subtitle']); ?></p>
                    <?php endif; ?>
                </header>
                
                <ul class="dcd-category-grid" role="list">
                    <?php foreach ($categories as $category) : ?>
                        <li class="dcd-category-item">
                            <a href="<?php echo esc_url(get_category_link($category->term_id)); ?>" class="dcd-category-link">
                                <span class="dcd-category-icon"><?php echo $this->get_category_icon($category->name); ?></span>
                                <span class="dcd-category-content">
                                    <span class="dcd-category-name"><?php echo esc_html($category->name); ?></span>
                                    <?php if ($settings['show_post_count']) : ?>
                                        <span class="dcd-category-count"><?php echo esc_html($category->count); ?> <?php echo ($category->count == 1) ? 'Post' : 'Posts'; ?></span>
                                    <?php endif; ?>
                                </span>
                                <span class="dcd-category-arrow">&#8594;</span>
                            </a>
                            <?php
                                $category_posts = $this->get_cached_category_posts($category->term_id);
                                $region_label = sprintf(
                                    esc_html__('Articles in %s', 'dynamic-category-display'),
                                    $category->name
                                );
                            ?>
                            <aside class="dcd-category-posts" role="region" aria-label="<?php echo esc_attr($region_label); ?>">
                                <?php if (!empty($category_posts)) : ?>
                                    <span class="dcd-category-posts__title"><?php esc_html_e('Articles', 'dynamic-category-display'); ?></span>
                                    <ul class="dcd-posts-list" role="list">
                                        <?php foreach ($category_posts as $category_post) : ?>
                                            <li class="dcd-posts-list__item">
                                                <a class="dcd-posts-list__link" href="<?php echo esc_url(get_permalink($category_post)); ?>">
                                                    <?php echo esc_html($this->format_post_title(get_the_title($category_post))); ?>
                                                </a>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php else : ?>
                                    <span class="dcd-category-posts__empty"><?php esc_html_e('No articles available yet.', 'dynamic-category-display'); ?></span>
                                <?php endif; ?>
                            </aside>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </article>
        </section>
        
        <?php
        $output = ob_get_clean();
        wp_cache_set($cache_key, $output, $this->cache_group, $this->render_cache_ttl);
        return $output;
    }
}

// Initialize plugin
new Dynamic_Category_Display();
